﻿namespace Microsoft.Samples.PlanMyNight.Data.Tests
{
    using System;
    using System.Linq;
    using System.Transactions;
    using Entities;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using System.Collections.Generic;

    [TestClass]
    public class ItinerariesRepositoryFixture
    {
        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void ShouldInsertNewItinerary()
        {
            var itinerary = new Itinerary() { Name = "My Test Itinerary", Created = DateTime.Now, UserId = Guid.NewGuid() };
            itinerary.Activities.Add(new ItineraryActivity()
            {
                ActivityId = "1",
                Order = 1,
                TypeId = 1,
                State = "WA",
                City = "Colville",
                Latitude = 48.545691,
                Longitude = -117.891798,
                Zip = "99114"
            });

            using (new TransactionScope())
            {
                var repository = new ItinerariesRepository();
                repository.Add(itinerary);
            }

            Assert.AreNotEqual(0, itinerary.Id);
        }

        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void ShouldSearchItinerariesUsingActivity()
        {
            using (new TransactionScope())
            {
                var data = new PlanMyNightEntities();
                foreach (var itinerary in data.Itineraries)
                {
                    data.DeleteObject(itinerary);
                }

                data.SaveChanges();

                var expected1 = new Itinerary { Name = "Match #1", IsPublic = true, Created = DateTime.Now };
                expected1.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "1",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                expected1.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "2",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                data.Itineraries.AddObject(expected1);
                data.SaveChanges();

                var expected2 = new Itinerary { Name = "Match #2", IsPublic = true, Created = DateTime.Now };
                expected2.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "3",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                expected2.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "1",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                data.Itineraries.AddObject(expected2);
                data.SaveChanges();

                var other = new Itinerary { Name = "Other", IsPublic = true, Created = DateTime.Now };
                other.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "3",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                other.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "4",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                data.Itineraries.AddObject(other);
                data.SaveChanges();

                var repo = new ItinerariesRepository();
                var result = repo.SearchByActivity("1", 10, 1);

                Assert.AreEqual(2, result.TotalItems);
                Assert.AreEqual(1, result.CurrentPage);
                Assert.IsTrue(result.Items.Any(a => a.Name == "Match #1"));
                Assert.IsTrue(result.Items.Any(a => a.Name == "Match #2"));
            }
        }

        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void ShouldSearchItinerariesByZipCode()
        {
            using (new TransactionScope())
            {
                var data = new PlanMyNightEntities();
                foreach (var itinerary in data.Itineraries)
                {
                    data.DeleteObject(itinerary);
                }

                data.SaveChanges();

                var expected1 = new Itinerary { Name = "Match #1", IsPublic = true, Created = DateTime.Now };
                expected1.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "1",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99113"
                });
                expected1.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "2",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                data.Itineraries.AddObject(expected1);
                data.SaveChanges();

                var expected2 = new Itinerary { Name = "Match #2", IsPublic = true, Created = DateTime.Now };
                expected2.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "3",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99112"
                });
                expected2.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "1",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99112"
                });
                data.Itineraries.AddObject(expected2);
                data.SaveChanges();

                var other = new Itinerary { Name = "Other", IsPublic = true, Created = DateTime.Now };
                other.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "3",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99112"
                });
                other.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "4",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99112"
                });
                data.Itineraries.AddObject(other);
                data.SaveChanges();

                var repo = new ItinerariesRepository();
                var result = repo.SearchByZipCode(1, "99112", 1, 2);

                Assert.AreEqual(2, result.TotalItems);
                Assert.AreEqual(2, result.CurrentPage);
                Assert.AreEqual(1, result.Items.Count);
            }
        }


        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void ShouldSearchItinerariesByRadius()
        {
            using (new TransactionScope())
            {
                var data = new PlanMyNightEntities();
                foreach (var itinerary in data.Itineraries)
                {
                    data.DeleteObject(itinerary);
                }

                data.SaveChanges();

                var expected1 = new Itinerary { Name = "Match #1", IsPublic = true, Created = DateTime.Now };
                expected1.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "1",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.54,
                    Longitude = -117.89,
                    Zip = "99114"
                });
                expected1.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "2",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.54,
                    Longitude = -117.89,
                    Zip = "99114"
                });
                data.Itineraries.AddObject(expected1);
                data.SaveChanges();

                var expected2 = new Itinerary { Name = "Match #2", IsPublic = true, Created = DateTime.Now };
                expected2.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "3",
                    TypeId = 2,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                expected2.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "1",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                data.Itineraries.AddObject(expected2);
                data.SaveChanges();

                var other = new Itinerary { Name = "Other", IsPublic = true, Created = DateTime.Now };
                other.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "3",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                other.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "4",
                    TypeId = 2,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                data.Itineraries.AddObject(other);
                data.SaveChanges();

                var repo = new ItinerariesRepository();
                var result = repo.SearchByRadius(1, -117.89179, 48.54569, 2, 1, 1);

                Assert.AreEqual(2, result.TotalItems);
                Assert.AreEqual(1, result.CurrentPage);
                Assert.AreEqual(1, result.Items.Count);
            }
        }
    }
}
